<?php

namespace App\Repositories;

use App\Models\Media;
use App\Models\User;
use Illuminate\Support\Facades\Storage;

class MediaRepository
{
        /**
     * Import original photo
     * 
     * @param string $url
     * @param array $additionlDataForImage
     * @return App\Models\Media
     */
    public static function importPhoto(string $url, $aditionalDataForImage = []): Media
    {
        $path_parts = pathinfo($url);

        $filename = 'temp-image.' . $path_parts['extension'];
        $tempImage = tempnam(sys_get_temp_dir(), $filename);
        // dd($url, $tempImage, $filename);
        copy($url, $tempImage);

        $imgInfo = getimagesize($tempImage);
        
        $file = new \Illuminate\Http\UploadedFile(
            $tempImage,
            rawurldecode($path_parts['basename']),
            $imgInfo['mime'],
            filesize($tempImage),
            true,
            TRUE
        );
        
        $folderName = now()->toDateString();
        $source = 'Alo.rs';
        if(isset($aditionalDataForImage) && !empty($aditionalDataForImage)){
            if(array_key_exists('article-date', $aditionalDataForImage)){
                $creationDate = $aditionalDataForImage['article-date'];
                $folderName = date('Y-m-d', strtotime($aditionalDataForImage['article-date']));
            }

            if(array_key_exists('source', $aditionalDataForImage)){
                $source = $aditionalDataForImage['source'];
            }

            if(array_key_exists('image_id', $aditionalDataForImage)){
                $imageId = $aditionalDataForImage['image_id'];
            }

            if(array_key_exists('image_alt', $aditionalDataForImage)){
                $imageAlt = $aditionalDataForImage['image_alt'];
            }
            if(array_key_exists('image_is_used', $aditionalDataForImage)){
                $imageIsUsed = $aditionalDataForImage['image_is_used'];
            }
        }

        $media = new Media();

        $media->creation_user = User::where('email', 'aleksandar.stanojevic@cubes.rs')->first()->id;
        $media->creation_date = $folderName;
        $media->source = $source;
        
        //insert created_at and updated_at 
        //if $additionalDataForImage['image-date'] is passed
        if(!empty($creationDate)){
            $media->created_at = $creationDate;
            $media->updated_at = $creationDate;
        }
        //insert imageId if $additionalDataForImage['image_id'] is passed
        if(!empty($imageId)){
            $media->id = $imageId;
        }

        //insert imageAlt if $additionalDataForImage['image_alt'] is passed
        if(!empty($imageAlt)){
            $media->alt = $imageAlt;
        }

        //insert imageId if $additionalDataForImage['image_is_used'] is passed
        if(!empty($imageIsUsed) || $imageIsUsed === false){
            $media->is_used = $imageIsUsed ? $imageIsUsed : 0;
        }
        
        $media->save();

        $result = self::saveImage($file, $media->id, $folderName);

        if($result === FALSE) {
            return JsonResource::make()->withError(__('Sliku nije bilo moguće snimiti'));
        }
        
        $media->update(['filename' => $result,]);

        return $media;
    }

    /**
     * Sacuvaj 'upload'-ovanu sliku na ciljnoj putanji:
     * /public/data/images/2020-07-23
     * 
     * @param \Illuminate\Http\UploadedFile $uploadedImage
     * @param integer $mediaId
     * 
     * @return string|FALSE
     */
    private static function saveImage(\Illuminate\Http\UploadedFile $uploadedImage, $mediaId, $folderName = null)
    {
        // formiraj putanju do slike
        $storeImagesAppPath = Storage::disk('public')->path('images');
        if(isset($folderName) && !empty($folderName)){
            $targetFolderName = $folderName;
        } else {
            $targetFolderName = now()->toDateString();
        }
        
        $targetFolderAbsolutePath = self::joinToPath($storeImagesAppPath, $targetFolderName);

        // kreiraj folder sa danasnjim datumom ukoliko vec ne postoji
        if( ! file_exists($targetFolderAbsolutePath) ) {
            try {
                mkdir($targetFolderAbsolutePath, 0755, TRUE);
            }
            catch (\ErrorException $e) {
                \Log::error('Greska: Prilikom kreiranja direktorijuma za sliku. '
                         . 'mkdir je vratio gresku. Putanja: ' . $targetFolderAbsolutePath
                         . ' Originalna greska: ' . $e->getMessage());
                return FALSE;
            }
        }


        // sacuvaj sliku na HDD-u
        $res = self::saveImageToHdd($uploadedImage, $targetFolderAbsolutePath, 
                                     $targetFolderName, $mediaId);
        
        return $res;
    }

    /**
     * Sacuvaj sliku na HDD sa 'template'-om imena:
     * id-stavke-originalno-ime-slike.jpg
     * 
     * @param UploadedFile $file         | 'file' objekat iz 'request'-a
     * @param string  $serverStoragePath | apsolutna putanja
     * @param string  $publicStoragePath | putanja gde je 'root' 'public'
     * @param integer $mediaId          | ID kategorije
     * @param string  $fnameSugar        | dodatak na kraju imena slike
     * 
     * @return string
     */
    private static function saveImageToHdd(
        \Illuminate\Http\UploadedFile $requestFile, 
        $serverStoragePath, 
        $publicStoragePath, 
        $mediaId, 
        $fnameSugar = ''
    ) {

        $file = \Intervention\Image\Facades\Image::make($requestFile);

        $fullFileName = self::constructImageFilename($requestFile, $mediaId, $fnameSugar);

        $file->save(self::joinToPath($serverStoragePath, $fullFileName));
        
        return self::joinToPath($publicStoragePath, $fullFileName);
    }

    private static function constructImageFilename($fileSrc, $imgId, $fnameSugar)
    {
        $templateStr = "%d_%s.%s";
        
        if($fileSrc instanceof \Symfony\Component\HttpFoundation\File\UploadedFile) {
            $file = $fileSrc;
            $filepath = $file->getClientOriginalName();
            $fileinfo = pathinfo($filepath);
        } elseif(is_string($fileSrc)) {
            $filepath = $fileSrc;
            $fileinfo = pathinfo($filepath);
        } else {
            throw new \InvalidArgumentException(
                "'\$addendum' parameter must be"
                . "of type string or UploadedFile"
            );
        }

        $fnameBase = \Str::slug($fileinfo['filename']);

        // empty($fnameSugar) ? : $fnameBase .= '_' . $fnameSugar; 
        $ext = $fileinfo['extension'] ?? '';
        $newFilename = sprintf($templateStr, $imgId, $fnameBase, $ext);
        
        return $newFilename;
    }

    /**
     * Vrati 'string' putanje od delova putanje.
     * 
     * param variadic(string) $parts
     * 
     * @return string
     */
    private static function joinToPath(...$parts)
    {
        $trimmedParts = array_map(function($el) {
            return rtrim($el, '/');
        }, $parts);

        return join('/', $trimmedParts);
    }
}