<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Resources\JsonResource;
use App\Http\Requests\Admin\WebsitesRequest;
use App\Models\Website\Website as Entity;
use Illuminate\Support\Facades\Validator;

/*
 * Kontroler za kreiranje i editovanje website-ova.
 */
class WebsitesController extends Controller
{
    /*
     * Vraca 'view' sa tabelom ovog modela
     */
    public function index()
    {
        return view('websites.index');
    }
    
    /*
     * Yajrabox datatables metod
     */
    public function datatable()
    {
        $query = Entity::query();

        return datatables($query)
            ->addColumn('actions', function ($entity) {
                return view('websites.partials.table.actions', 
                            ['entity' => $entity]);
            })
            ->addColumn('icon', function ($entity) {
                return view('websites.partials.table.icon', 
                            ['entity' => $entity]);
            })
            ->editColumn('active', function ($entity) {
                return view('websites.partials.table.active', ['entity' => $entity]);
            })
            ->rawColumns(['actions', 'icon', 'active'])
            ->setRowAttr([
                'data-id' => function($entity) {
                    return $entity->id;
                }
            ])
            ->make(true);
    }
    
    /*
     * Prikazi 'view' za kreiranje novog website-a
     */
    public function create()
    {
        $entity = new Entity();
        
        return view('websites.create', [
            'entity' => $entity
        ]);
    }
    
    /**
     * @param WebsiteRequest $request | 'custom request' objekat u kome
     *                                           se nalazi validacioni 'array'.
     * 
     * Snimi podatke novog website-a u bazu podataka.
     */
    public function store(WebsitesRequest $request)
    {
        $notificationMessage = __("Website je sačuvan");
        
        $data = $request->validated();
        unset($data['image_logo']);
        unset($data['image_icon']);

        $entity = Entity::create($data);
        
        $entity->save();
        
        $entity->storeImage('image_logo');
        $entity->storeImage('image_icon');

        if ($request->wantsJson()) {
            return JsonResource::make()->withSuccess($notificationMessage);
        }
        
        return redirect()->route('websites.index')
                         ->withSystemMessage($notificationMessage);
    }
    
    /*
     * Prikazi 'view' za izmenu podataka website-a.
     */
    public function edit(Entity $entity)
    {

        return view('websites.edit', [
            'entity' => $entity
        ]);
    }
    
    /**
     * @param WebsiteRequest $request | 'custom request' objekat u kome
     *                                           se nalazi validacioni 'array'.
     * 
     * Snimi izmenjene podatke Website-a u bazu podataka.
     */
    public function update(WebsitesRequest $request, Entity $entity)
    {   
        $notificationMessage = __("Website je izmenjen");
        
        $data = $request->validated();
        
        unset($data['image_logo'], $data['image_icon'], $data['title']);
        
        $entity->fill($data);
        
        $entity->save();
        
        $entity->updateImage('image_logo');
        $entity->updateImage('image_icon');

        if ($request->wantsJson()) {
            return JsonResource::make($entity)->withSuccess($notificationMessage);
        }
        
        return redirect()->route('websites.index')
                         ->withSystemMessage($notificationMessage);
    }
    
    public function changeActive(Entity $entity)
    {   
        $request = $this->request;
        
        $entity->active = $entity->active == 1 ? 0 : 1;
        
        $entity->save();
        
        $status = $entity->isActive() ? 'aktiviran' : 'deaktiviran';
        
        if ($request->wantsJson()) {
            return JsonResource::make()->withSuccess(__('Website je ' . $status . '!'));
        }
        
        return redirect()->route('entities.list')->withSystemSuccess(__('Website je ' . $status . '!'));
    }
    
    /*
     * Izbrisi website iz baze podataka
     */
    public function obrisi(Entity $entity)
    {   
        $entity->delete();
        
        $notificationMessage = __('Website je obrisan`');

        if (request()->wantsJson()) {
            return JsonResource::make($entity)
                               ->withSuccess($notificationMessage);
        }
        
        return redirect()->route('websites.index')
                         ->withSystemMessage($notificationMessage);
    }
    
    /*
     * Obrisi sliku preko ajaks poziva
     */
    public function ajaxDeleteImage(Request $request, Entity $entity)
    {
        $request->validate([
            "imageClass" => ["required", "string", "exists:images,class"]
        ]);
        
        $stat = $entity->deleteImage($request->imageClass);
        if(!$stat) {
            return JsonResource::make()->withError(__("Greška. Slika nije mogla biti izbrisana."));
        }
        
        return JsonResource::make()->withSuccess(__("Slika je izbrisana."));
    }
    
}
