<?php

/**
 * Class
 *
 * PHP version 7.2
 *
 * @category   class
 * @copyright  Cubes d.o.o.
 * @license    GPL http://opensource.org/licenses/gpl-license.php GNU Public License
 */

namespace App\Models;

use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;
use App\Models\Utils\StoreFilesModel;
use App\Models\Utils\ActiveInactive;
use App\Models\Utils\ActiveInactiveModel;
use App\Models\Website\Website;
use Junges\ACL\Concerns\HasGroups;


class User extends Authenticatable implements ActiveInactive
{
    use Notifiable, StoreFilesModel, ActiveInactiveModel;
    use \App\Models\Utils\ImageableTrait;
    use HasGroups;

    // constants used in method getFullName() as modes for full-name generation
    const FN_URL = "url";
    const FN_FILENAME = "filename";

    /*
     * ID aplikacionog sistemski nedostupnog korisnika.
     */
    const APPLICATION_USER_ID = 0;

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'role_id', 'active', 'first_name', 'last_name', 'email', 'maticna_pj', 'password', 'phone'
    ];

    /**
     * Attributes that should be hidden for arrays
     *
     * @var array
     */
    protected $hidden = [
        'password', 'remember_token',
    ];

    protected $imageResizeRecepies = [
        "slika_profil" => "avatar"
    ];  

    /**
     * Override default delete because of the
     */
    public function delete()
    {
        $this->deleteFile('photo');
        parent::delete();
    }

    /**
     * spectial function which determinates wether user is sysadmin
     */
    public function isSystemAdmin()
    {
        return false;
    }
    
    
    /**
     * Get user's full name.
     * 
     * @param  string  $mode - mode in which to construct fullname 
     * @return string
     */
    public function getFullName($mode = NULL)
    {
        if($mode == self::FN_URL) {
            return \Str::slug($this->first_name . "-" . $this->last_name);
        }
        else if($mode == self::FN_FILENAME) {
            return \Str::slug($this->first_name . "_" . $this->last_name);
        }
        
        return $this->first_name . " " . $this->last_name;
    }
    
    /**
     * @return Website[] Kolekcija website-ova
     */
    public function getWebsites()
    {
        return $this->websites;
    }

    /**
     * Vraca maticni website ako ima privilegije
     * @return Website | null
     */
    public function getPrimaryWebsite()
    {
        return $this->primaryWebsite;
    }

    /*
     * Vrati 'public URL' do 'logo' slike - ukoliko website ima 
     * 'logo' sliku.
     */
    public function getProfileImg()
    {
        $logoImg = $this->getImage('slika_profil');
        return !empty($logoImg) ? $logoImg->getUrl() : config('newscms.default_profile_img_src');
    }
    
    public function isAdmin()
    {
        return false;
    }
    
    public function websites()
    {
        return $this->belongsToMany(
                Website::class, 
                'users_websites',
                'user_id',
                'website_id'
            )->active();
    }

    public function primaryWebsite()
    {
        return $this->belongsTo(Website::class, 'primary_website')
                    ->active();
    }
}
